#include <ansi_c.h>
#include <utility.h>
#include <formatio.h>
#include <stdlib.h>
#include <userint.h>
#include "gmsspx.h"
#include "gmsspxu.h"

/*= Global Variable Definitions =============================================*/

long panelHandle[5];
unsigned long instrHandle;

/*= Message Arrays ==========================================================*/
static char *initCtrlsHelpStr[] = {
									"This control specifies the interface and address of the device\n"
									"that is to be initialized (Interface Descriptor). The exact\n"
									"grammar to be used in this control is shown in the note below.\n\n"
									"Default Value:  \"GPIB::13\"\n\n"

									"Interface   Grammar\n"
									"------------------------------------------------------\n"
									"GPIB        GPIB[board]::primary address[::secondary address]\n"
									"            [::INSTR]\n"
									"Serial      ASRL<port>::INSTR\n\n"
									"The ASRL keyword is used with Serial instruments.\n"
									"The GPIB keyword is used with GPIB instruments.\n",



                                    "This control specifies if an ID Query is sent to the\n"
                                    "instrument during the initialization procedure.\n\n"
                                    "Valid Range:\n"
                                    "0 - Skip Query\n"
                                    "1 - Do Query (Default Value)",
                                    
                                    "This control specifies if the instrument is to be reset to its\n"
                                    "power-on settings during the initialization procedure.\n\n"
                                    "Valid Range:\n"
                                    "0 - Don't Reset\n"
                                    "1 - Reset Device (Default Value)"};  
        
static char *configCtrlsHelpStr[] = {
    "This control returns output setpoint value\n" 
    "of the output current.",
                                    
    "This control returns output setpoint value\n" 
    "of the output voltage.",
                                    
    "This control sets the setpoint value of the output current.\n\n"   
    "Default Value: 1.0\n\n"
    "Valid Range:\n"
    "--------------------------------------------------------------------------------\n"
    "Model              Valid Range (A)      Resolution (mA)   \n"
    "--------------------------------------------------------------------------------\n"
    "32 N 20 RU 10 P        0...10 A                1 mA       \n"
    "32 N 40 RU  6 P         0...6 A                  1 mA       \n"
    "32 N 80 RU  3 P         0...3 A                  1 mA       \n"
    "32 N 20 RU 20 P        0...20 A                1 mA       \n"
    "32 N 40 RU 12 P        0...12 A                1 mA       \n"
    "32 N 80 RU  6 P         0...6 A                  1 mA       \n"
    "32 N 32 RU 18 P        0...18 A                1 mA       \n"
    "62 N 40 RU 25 P        0...25 A                1 mA       \n"
    "62 N 40 RU 50 P        0...50 A                1 mA       \n"
    "64 N 40 RU100 P       0...100 A              1 mA       \n"
    "64 N 40 RU150 P       0...150 A              1 mA       \n"
    "62 N 80 RU12.5P       0...12.5 A             1 mA       \n"
    "62 N 80 RU 25 P        0...25 A              1 mA       \n"
    "64 N 80 RU 50 P        0...50 A              1 mA       \n"
    "64 N 80 RU 75 P        0...75 A              1 mA       \n"
    "62 N 52 RU 25 P        0...25 A              1 mA       \n"
    "62 N 52 RU 50 P        0...50 A              1 mA       \n"
    "64 N 52 RU 100P       0..100 A              1 mA       \n"
    "64 N 52 RU 150P       0..150 A              1 mA       \n"
    "-----------------------------------------------------------------------------\n\n"
    "Note:\n"
    "(1) Value must be smaler than current limit.",
                                    
    "This control sets the setpoint value of the output voltage.\n\n"
    "Default Value: 10.0\n\n"
    "Valid Range:\n"
    "-----------------------------------------------------------------------------\n"
    "Model              Valid Range (V)      Resolution (mV)   \n"
    "-----------------------------------------------------------------------------\n"
    "32 N 20 RU 10 P        0...20 V                5 mV       \n"
    "32 N 40 RU  6 P         0...40 V               10 mV       \n"
    "32 N 80 RU  3 P         0...80 V               20 mV       \n"
    "32 N 20 RU 20 P        0...20 V                5 mV       \n"
    "32 N 40 RU 12 P        0...40 V               10 mV       \n"
    "32 N 80 RU  6 P         0...80 V               20 mV       \n"
    "32 N 32 RU 18 P        0...32 V               10 mV       \n"
    "62 N 40 RU 25 P        0...40 V               10 mV       \n"
    "62 N 40 RU 50 P        0...40 V               10 mV       \n"
    "64 N 40 RU100 P       0...40 V               10 mV       \n"
    "64 N 40 RU150 P       0...40 V               10 mV       \n"
    "62 N 80 RU12.5P       0...80 V               20 mV       \n"
    "62 N 80 RU 25 P        0...80 V               20 mV       \n"
    "64 N 80 RU 50 P        0...80 V               20 mV       \n"
    "64 N 80 RU 75 P        0...80 V               20 mV       \n"
    "62 N 52 RU 25 P        0...52 V               20 mV       \n"
    "62 N 52 RU 50 P        0...52 V               20 mV       \n"
    "64 N 52 RU 100P       0...52 V               20 mV       \n"
    "64 N 52 RU 150P       0...52 V               20 mV       \n"
    "-----------------------------------------------------------------------------\n\n"
    "Note:\n"
    "(1) Value must be smaller than voltage limit.",

    "This control defines the upper soft limit for the current\n"
    "setpoint, thus enabling you to prevent the output current from\n"
    "being set in excess of a certain value.\n\n"
    "Default Value: 3.0\n\n"
    "Valid Range:\n"
    "--------------------------------------------------------------------------------\n"
    "Model              Valid Range (A)      Resolution (mA)   \n"
    "--------------------------------------------------------------------------------\n"
    "32 N 20 RU 10 P        0...10 A                1 mA       \n"
    "32 N 40 RU  6 P         0...6 A                  1 mA       \n"
    "32 N 80 RU  3 P         0...3 A                  1 mA       \n"
    "32 N 20 RU 20 P        0...20 A                1 mA       \n"
    "32 N 40 RU 12 P        0...12 A                1 mA       \n"
    "32 N 80 RU  6 P         0...6 A                  1 mA       \n"
    "32 N 32 RU 18 P        0...18 A                1 mA       \n"
    "62 N 40 RU 25 P        0...25 A                1 mA       \n"
    "62 N 40 RU 50 P        0...50 A                1 mA       \n"
    "64 N 40 RU100 P       0...100 A              1 mA       \n"
    "64 N 40 RU150 P       0...150 A              1 mA       \n"
    "62 N 80 RU12.5P       0...12.5 A             1 mA       \n"
    "62 N 80 RU 25 P        0...25 A                1 mA       \n"
    "64 N 80 RU 50 P        0...50 A                1 mA       \n"
    "64 N 80 RU 75 P        0...75 A                1 mA       \n"
    "62 N 52 RU 25 P        0...25 A                1 mA       \n"
    "62 N 52 RU 50 P        0...50 A                1 mA       \n"
    "64 N 52 RU 100P       0..100 A                1 mA       \n"
    "64 N 52 RU 150P       0..150 A                1 mA       \n"
    "-----------------------------------------------------------------------------\n\n"
    "Note:\n"
    "(1) Value must be larger than current setpoint.",
                                    
    "This control defines the upper soft limit for the voltage\n"
    "setpoint, thus enabling you to prevent the output voltage from\n"
    "being set in excess of a certain value.\n\n"
    "Default Value: 20.0\n\n"
    "Valid Range:\n"
    "-----------------------------------------------------------------------------\n"
    "Model              Valid Range (V)      Resolution (mV)   \n"
    "-----------------------------------------------------------------------------\n"
    "32 N 20 RU 10 P        0...20 V                5 mV       \n"
    "32 N 40 RU  6 P         0...40 V               10 mV       \n"
    "32 N 80 RU  3 P         0...80 V               20 mV       \n"
    "32 N 20 RU 20 P        0...20 V                5 mV       \n"
    "32 N 40 RU 12 P        0...40 V               10 mV       \n"
    "32 N 80 RU  6 P         0...80 V               20 mV       \n"
    "32 N 32 RU 18 P        0...32 V               10 mV       \n"
    "62 N 40 RU 25 P        0...40 V               10 mV       \n"
    "62 N 40 RU 50 P        0...40 V               10 mV       \n"
    "64 N 40 RU100 P       0...40 V               10 mV       \n"
    "64 N 40 RU150 P       0...40 V               10 mV       \n"
    "62 N 80 RU12.5P       0...80 V               20 mV       \n"
    "62 N 80 RU 25 P        0...80 V               20 mV       \n"
    "64 N 80 RU 50 P        0...80 V               20 mV       \n"
    "64 N 80 RU 75 P        0...80 V               20 mV       \n"
    "62 N 52 RU 25 P        0...52 V               20 mV       \n"
    "62 N 52 RU 50 P        0...52 V               20 mV       \n"
    "64 N 52 RU 100P       0...52 V               20 mV       \n"
    "64 N 52 RU 150P       0...52 V               20 mV       \n"
    "-----------------------------------------------------------------------------\n\n"
    "Note:\n"
    "(1) Value must be larger than voltage setpoint.",         
};
                                    
static char backgroundPnlHelpStr[] = {"This is a simple demo program using\n"
                                    "the Gossen Metrawatt Konstanter SSP X Series\n"
                                    "instrument driver provided in the package.\n"};
                           
static char initPnlHelpStr[] =     {"This panel performs the following initialization actions:\n\n"
                                    "- Opens a instrument session.\n\n"
                                    "- Performs an identification query on the Instrument.\n\n"
                                    "- Sends initialization commands to the instrument that set any\n"
                                    "necessary programmatic variables to the state necessary\n"
                                    "for the operation of the instrument driver."};

static char configPnlHelpStr[] = {"This panel configures the instrument."};

/*= Main Function ===========================================================*/

void main ()
{
    // Load panels.
    panelHandle[BCKGRND] = LoadPanel (panelHandle[BCKGRND], "gmsspxu.uir", BCKGRND);
    panelHandle[CONFIG] = LoadPanel (panelHandle[BCKGRND], "gmsspxu.uir", CONFIG); 
    panelHandle[INIT] = LoadPanel (panelHandle[BCKGRND], "gmsspxu.uir", INIT);
    // Display panels.
    DisplayPanel (panelHandle[BCKGRND]);
    DisplayPanel (panelHandle[INIT]);
    SetSleepPolicy (VAL_SLEEP_MORE);
    RunUserInterface ();
}   

/*===========================================================================*/
/* Function: Initialize Instrument                                           */
/* Purpose:  This is a callback function of the Continue button on the       */
/*           Initialize panel. It initializes the instrument and switches to */
/*           the panel Configure.                                            */
/*===========================================================================*/
int CVICALLBACK initInstrument (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    ViStatus error;
    ViChar instrDescr[256];
    ViInt16  rst, id;
    ViChar addr[16];
    char errMsg[256], errBuffer[256];

    switch (event) {
        case EVENT_COMMIT:
            SetWaitCursor(1);
            GetCtrlVal (panelHandle[INIT], INIT_ADDRESS, addr);
            GetCtrlVal (panelHandle[INIT], INIT_ID, &id);
            GetCtrlVal (panelHandle[INIT], INIT_RST, &rst);
            Fmt (instrDescr, "%s", addr);
            if ((error = gmsspx_init (instrDescr, id, rst, &instrHandle)) < 0 ) {
                gmsspx_error_message(VI_NULL, error, errMsg);
                sprintf (errBuffer, "Initialization Error:\n\n%s\n\nCheck your "
                        "connections and make sure you have the right GPIB address.",
                         errMsg);
                MessagePopup ("ERROR!", errBuffer);
                SetWaitCursor(0);
                return(0);   
            }
            HidePanel (panelHandle[INIT]);
            DisplayPanel (panelHandle[CONFIG]);
            SetWaitCursor(0); 
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button causes the instrument to be initialized.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Configure Instrument                                            */
/* Purpose:  This is a callback function of the Continue button on the       */
/*           Configure panel .  					                         */
/*===========================================================================*/
int CVICALLBACK config (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    ViInt32 errorCode = 0;
    ViStatus error;
    char errMsg[256], errorMessage[256], errBuffer[256];
    double volt, curr, volt_lim, curr_lim, nom_volt, nom_curr;

    switch (event) {
        case EVENT_COMMIT:
            GetCtrlVal (panelHandle[CONFIG], CONFIG_VOLT, &volt);
            GetCtrlVal (panelHandle[CONFIG], CONFIG_CURR, &curr);
            GetCtrlVal (panelHandle[CONFIG], CONFIG_VOLT_LIMIT, &volt_lim);
            GetCtrlVal (panelHandle[CONFIG], CONFIG_CURR_LIMIT, &curr_lim);
            SetWaitCursor(1);
            error = gmsspx_AppExample (instrHandle, volt_lim, curr_lim, volt, curr,
                                    &nom_volt, &nom_curr);
            SetWaitCursor(0);
            if (error < 0) {
                gmsspx_error_message (VI_NULL, error, errMsg);
                gmsspx_error_query (instrHandle, &errorCode, errorMessage);
                sprintf (errBuffer, "Error Message: %s \n\nInstrument Specific Error: %d: %s",
                    errMsg, errorCode, errorMessage);
                MessagePopup ("Error:", errBuffer);
            }
            else {
                SetCtrlVal (panelHandle[CONFIG], CONFIG_VOLT_NOM, nom_volt);
                SetCtrlVal (panelHandle[CONFIG], CONFIG_CURR_NOM, nom_curr);
            }
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button will configure the instrument and take its settings.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Cancel                                                          */
/* Purpose:  Called by the init panel this function pops up a confirmation   */
/*           dialog box and then quits the user interface, if desired.       */
/*===========================================================================*/
int CVICALLBACK Cancel (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    SetWaitCursor(0);
    switch (event) {
        case EVENT_COMMIT:
            if ((ConfirmPopup ("Exit Application", 
                   "Are you sure you want to quit this application?")) == 1) {
                QuitUserInterface (0);
            }
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Control Help", "Closes the application.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Control Help                                                    */
/* Purpose:  This is a callback function of all controls that configure the  */
/*           instrument. On the right mouse-click on the control a help      */
/*           window describing its purpose is displayed.                     */
/*===========================================================================*/
int CVICALLBACK controlHelp (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{   
    SetWaitCursor(0);
    if (event == EVENT_RIGHT_CLICK) {
        if (panel == panelHandle[INIT]) 
            MessagePopup ("Help", initCtrlsHelpStr[control-4]);
        if (panel == panelHandle[CONFIG]) 
            MessagePopup ("Help", configCtrlsHelpStr[control-4]);
    }
    return 0;
}

/*===========================================================================*/
/* Function: Panel Help                                                      */
/* Purpose:  This is a callback function of the menu bar. It displays a help */
/*           window describing panel being used.                             */
/*===========================================================================*/
void CVICALLBACK panelHelp (int menuBar, int menuItem, void *callbackData,
        int panel)
{
    SetWaitCursor(0);
    if (panel == panelHandle[BCKGRND])
        MessagePopup ("Help", backgroundPnlHelpStr);
    if (panel == panelHandle[INIT])
        MessagePopup ("Help", initPnlHelpStr);
    if (panel == panelHandle[CONFIG])
        MessagePopup ("Help", configPnlHelpStr);
    return;
}

/*===========================================================================*/
/* Function: Launch Close Panel                                              */
/* Purpose:  This is a callback function of the button Close on the          */
/*           panel Configure. It pops-up a panel that will close the         */
/*           instrument.                                                     */
/*===========================================================================*/
int CVICALLBACK launchClose (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    
    SetWaitCursor(0);
    switch (event) {
        case EVENT_COMMIT:                                       
            if ((ConfirmPopup ("Exit Application", "Are you sure you want to quit "
                    "this application?")) == 1) 
            {
                gmsspx_close (instrHandle);
                QuitUserInterface (0);
                exit (-1);
            }
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button will pop-up a panel to close the instrument.");
            break;
    }
    return 0;
}   

/*= End =====================================================================*/
